"use client";
import Head from "next/head";
import Link from "next/link";
import { useSession } from "next-auth/react";
import { useLanguage } from "@/contexts/LanguageContext";
import { useEffect, useState } from "react";
import {
  Plus,
  Ticket,
  Users,
  Wrench,
  Activity,
  ShieldCheck,
  Hospital,
  AlertTriangle,
  ClipboardList,
  Loader2,
} from "lucide-react";

interface DashboardStats {
  openTickets: number;
  activeTechnicians: number;
  pendingMaintenance: number;
  totalTickets: number;
  recentTickets: any[];
}

export default function Home() {
  const { data: session, status } = useSession();
  const { t } = useLanguage();
  const email = session?.user?.email;
  const role = (session?.user as any)?.role?.toUpperCase?.() || "USER";
  const isAdmin = role === "ADMIN";

  const [stats, setStats] = useState<DashboardStats>({
    openTickets: 0,
    activeTechnicians: 0,
    pendingMaintenance: 0,
    totalTickets: 0,
    recentTickets: [],
  });
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchDashboardData = async () => {
      try {
        setLoading(true);

        // Fetch tickets data
        const ticketsResponse = await fetch("/api/tickets");
        const ticketsData = await ticketsResponse.json();

        if (ticketsData.ok) {
          const tickets = ticketsData.tickets;
          const openTickets = tickets.filter((t: any) => !t.closedAt).length;
          const totalTickets = tickets.length;
          const recentTickets = tickets.slice(0, 5); // Last 5 tickets

          setStats((prev) => ({
            ...prev,
            openTickets,
            totalTickets,
            recentTickets,
          }));
        }

        // Fetch technicians data (admin only)
        if (isAdmin) {
          const techniciansResponse = await fetch(
            "/api/admin/users?role=technician"
          );
          const techniciansData = await techniciansResponse.json();

          if (techniciansData.ok) {
            setStats((prev) => ({
              ...prev,
              activeTechnicians: techniciansData.users.length,
            }));
          }
        }

        // Simulate pending maintenance (could be fetched from a real API)
        setStats((prev) => ({
          ...prev,
          pendingMaintenance: Math.floor(Math.random() * 10) + 1,
        }));
      } catch (error) {
        console.error("Error fetching dashboard data:", error);
      } finally {
        setLoading(false);
      }
    };

    if (session) {
      fetchDashboardData();
    }
  }, [session, isAdmin]);

  // Show welcome page for unauthenticated users
  if (!session && status !== "loading") {
    return (
      <>
        <Head>
          <title>{t("nav.hospitalPortal")}</title>
        </Head>

        <main className='max-w-4xl mx-auto px-4 py-12 text-center'>
          <div className='flex justify-center mb-6'>
            <div className='w-20 h-20 bg-primary rounded-lg flex items-center justify-center'>
              <Hospital className='w-10 h-10 text-primary-fore' />
            </div>
          </div>

          <h1 className='text-4xl font-bold text-foreground mb-4'>
            {t("dashboard.welcome")}
          </h1>
          <p className='text-lg text-muted-foreground mb-8'>
            {t("dashboard.subtitle")}
          </p>

          <div className='flex flex-col sm:flex-row gap-4 justify-center items-center'>
            <Link
              href='/login'
              className='inline-flex items-center justify-center gap-2 px-6 py-3 bg-primary text-white rounded-lg hover:bg-primary/90 transition-colors font-medium'
            >
              {t("dashboard.signIn")}
            </Link>
            <Link
              href='/signup'
              className='inline-flex items-center justify-center gap-2 px-6 py-3 bg-secondary text-secondary-foreground rounded-lg hover:bg-secondary/90 transition-colors text-base font-medium'
            >
              {t("dashboard.createAccount")}
            </Link>
          </div>

          <div className='mt-16 grid sm:grid-cols-3 gap-6'>
            <Link
              href='/tickets/new'
              className='p-6 rounded-lg border bg-card hover:bg-accent transition-colors cursor-pointer block'
            >
              <Ticket className='w-8 h-8 text-primary mx-auto mb-3' />
              <h3 className='font-semibold text-foreground mb-2'>
                {t("dashboard.features.createTickets.title")}
              </h3>
              <p className='text-sm text-muted-foreground'>
                {t("dashboard.features.createTickets.description")}
              </p>
            </Link>
            <div className='p-6 rounded-lg border bg-card'>
              <Activity className='w-8 h-8 text-primary mx-auto mb-3' />
              <h3 className='font-semibold text-foreground mb-2'>
                {t("dashboard.features.trackStatus.title")}
              </h3>
              <p className='text-sm text-muted-foreground'>
                {t("dashboard.features.trackStatus.description")}
              </p>
            </div>
            <div className='p-6 rounded-lg border bg-card'>
              <ShieldCheck className='w-8 h-8 text-primary mx-auto mb-3' />
              <h3 className='font-semibold text-foreground mb-2'>
                {t("dashboard.features.secureSystem.title")}
              </h3>
              <p className='text-sm text-muted-foreground'>
                {t("dashboard.features.secureSystem.description")}
              </p>
            </div>
          </div>
        </main>
      </>
    );
  }

  return (
    <>
      <Head>
        <title>Hospital Portal — Dashboard</title>
      </Head>

      <main className='max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 space-y-10'>
        {/* Header */}
        <header className='flex items-center justify-between'>
          <div className='flex items-center gap-3'>
            <Hospital className='w-8 h-8 text-primary' />
            <div>
              <h1 className='text-3xl font-bold text-foreground'>
                {t("dashboard.title")}
              </h1>
              {email && (
                <p className='text-sm text-muted-foreground mt-1'>
                  {t("dashboard.signedInAs")}{" "}
                  <b className='text-foreground'>{email}</b> ({role})
                </p>
              )}
            </div>
          </div>
        </header>

        {/* Overview Stats */}
        <section className='grid sm:grid-cols-2 lg:grid-cols-4 gap-6'>
          {loading ? (
            Array.from({ length: 4 }).map((_, i) => (
              <div key={i} className='rounded-lg border p-5 bg-card shadow-sm'>
                <div className='flex items-center justify-between'>
                  <div className='space-y-2'>
                    <div className='h-4 bg-muted rounded animate-pulse'></div>
                    <div className='h-8 bg-muted rounded animate-pulse'></div>
                  </div>
                  <Loader2 className='w-6 h-6 text-muted-foreground animate-spin' />
                </div>
              </div>
            ))
          ) : (
            <>
              <DashboardCard
                title={t("dashboard.stats.openTickets")}
                value={stats.openTickets.toString()}
                icon={<Ticket className='w-6 h-6 text-primary' />}
              />
              <DashboardCard
                title={t("dashboard.stats.activeTechnicians")}
                value={stats.activeTechnicians.toString()}
                icon={<Users className='w-6 h-6 text-primary' />}
              />
              <DashboardCard
                title={t("dashboard.stats.pendingMaintenance")}
                value={stats.pendingMaintenance.toString()}
                icon={<Wrench className='w-6 h-6 text-primary' />}
              />
              <DashboardCard
                title={t("dashboard.stats.totalTickets")}
                value={stats.totalTickets.toString()}
                icon={<ShieldCheck className='w-6 h-6 text-primary' />}
                valueColor='text-blue-600'
              />
            </>
          )}
        </section>

        {/* Quick Actions */}
        <section>
          <h2 className='text-lg font-semibold mb-4 text-foreground flex items-center gap-2'>
            <ClipboardList className='w-5 h-5 text-primary' />
            {t("common.quickActions") || "Quick Actions"}
          </h2>
          <div className='flex flex-wrap gap-4'>
            <Link
              href='/tickets/new'
              className='inline-flex items-center justify-center gap-2 px-5 py-2 bg-primary text-white rounded-lg hover:bg-primary/90 transition-colors'
            >
              <Plus className='w-4 h-4' />
              {t("sidebar.newTicket")}
            </Link>
            <Link
              href='/tickets'
              className='inline-flex items-center justify-center gap-2 px-5 py-2 bg-secondary text-secondary-foreground rounded-lg hover:bg-secondary/90 transition-colors'
            >
              <Ticket className='w-4 h-4' />
              {t("dashboard.viewAll")}
            </Link>
            {isAdmin && (
              <Link
                href='/admin/technicians'
                className='inline-flex items-center justify-center gap-2 px-5 py-2 bg-secondary text-secondary-foreground rounded-lg hover:bg-secondary/90 transition-colors'
              >
                <Users className='w-4 h-4' />
                {t("admin.manageTechnicians") || "Manage Technicians"}
              </Link>
            )}
          </div>
        </section>

        {/* Recent Activity */}
        <section className='rounded-lg border p-6 bg-card shadow-sm'>
          <div className='flex items-start gap-3'>
            <Activity className='w-6 h-6 text-primary mt-1' />
            <div className='flex-1'>
              <h3 className='text-lg font-semibold text-foreground mb-4'>
                {t("dashboard.recentTickets")}
              </h3>
              {loading ? (
                <div className='space-y-3'>
                  {Array.from({ length: 3 }).map((_, i) => (
                    <div key={i} className='flex items-center gap-3'>
                      <div className='w-2 h-2 bg-muted rounded-full animate-pulse'></div>
                      <div className='flex-1 space-y-1'>
                        <div className='h-4 bg-muted rounded animate-pulse'></div>
                        <div className='h-3 bg-muted rounded animate-pulse w-1/2'></div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : stats.recentTickets.length > 0 ? (
                <div className='space-y-3'>
                  {stats.recentTickets.slice(0, 5).map((ticket: any) => (
                    <Link
                      key={ticket.id}
                      href={`/tickets/${ticket.id}`}
                      className='flex items-center gap-3 p-3 rounded-lg bg-muted/30 hover:bg-muted/50 transition-colors'
                    >
                      <div className='w-2 h-2 bg-primary rounded-full'></div>
                      <div className='flex-1 min-w-0'>
                        <p className='text-sm font-medium text-foreground truncate'>
                          {ticket.title}
                        </p>
                        <p className='text-xs text-muted-foreground'>
                          {new Date(ticket.createdAt).toLocaleDateString()} •{" "}
                          {ticket.department}
                        </p>
                      </div>
                    </Link>
                  ))}
                </div>
              ) : (
                <div className='text-center py-8 text-muted-foreground'>
                  <Ticket className='w-12 h-12 mx-auto mb-4 opacity-50' />
                  <p>{t("tickets.noTickets")}</p>
                  <p className='text-sm mt-1'>
                    {t("dashboard.getStarted") ||
                      "Create your first ticket to get started"}
                  </p>
                </div>
              )}
            </div>
          </div>
        </section>
      </main>
    </>
  );
}

/* --- Small reusable card component --- */
function DashboardCard({
  title,
  value,
  icon,
  valueColor,
}: {
  title: string;
  value: string;
  icon: React.ReactNode;
  valueColor?: string;
}) {
  return (
    <div className='rounded-lg border p-5 bg-card shadow-sm'>
      <div className='flex items-center justify-between'>
        <div>
          <p className='text-sm text-muted-foreground'>{title}</p>
          <h3
            className={`text-2xl font-semibold text-foreground ${
              valueColor || ""
            }`}
          >
            {value}
          </h3>
        </div>
        {icon}
      </div>
    </div>
  );
}
